// HeadTurner behavior

define(
  function () {
	"use strict";
	
	var kNumHorizontalLayerParams = 5;	// tied to IDs layer0 - layer4

	// returns array of non-empty layer-arrays (up to 5, each of which refers to one or more layers)
	function getActiveLayerArrays(args) {
		var aaLayers = [], i;
		
		for (i = 0; i < kNumHorizontalLayerParams; ++i) {
			var aLayers = args.getStaticParam("layer"+i);
			if (aLayers.length > 0) {
				aaLayers.push(aLayers);
			}
		}
		
		return aaLayers;
	}
	  
	
	// returns index into segment given table of segment boundaries (between 0 and table.length inclusive)
	function chooseBoundary(table, v) {
		for (var i=0; i<table.length; ++i) {
			if (v < table[i]) {
				return i;
			}
		}
		
		return table.length;
	}

	  
	// because head up/down angling looks good on quarter & profile views, but not front
	//	we only trigger the up/down views when the head is nearly facing front
	function chooseLayerBasedOnAngle(self, args, headAngle, upDownAngle) {
		/* angle segmentation visualization
				# of segments	-1.0 ----------- 0 ----------- +1.0
						two						 |
						three				| 		  |
						four			  | 	 | 		|
						five			|      |   |      |
		*/
		var aLayersToShow,
			table = {	2: [0],
						3: [-0.20, +0.20],
						4: [-0.25, 0, +0.25],
						5: [-0.30, -0.15, +0.15, +0.30]
					},
			numViews = self.aaLayers.length,
			bInFrontalSweetSpot = Math.abs(headAngle) < 0.15;	// arbitrary threshold, but does match the 5-table middle boundaries
		
		//console.logToUser("upDownAngle = " + upDownAngle + " headAngle = " + headAngle);
		if (bInFrontalSweetSpot) {
			if (upDownAngle < -0.12 && self.aUpwardLayers.length > 0) {
				aLayersToShow = self.aUpwardLayers;
			} else if (upDownAngle > 0.20 && self.aDownwardLayers.length > 0) {
				aLayersToShow = self.aDownwardLayers;
			}
		}
		
		if (!aLayersToShow && numViews > 1) {
			var indexToShow = chooseBoundary(table[numViews], headAngle);
			aLayersToShow = self.aaLayers[indexToShow];
		}
		
		if (!aLayersToShow && numViews > 0) {
			aLayersToShow = self.aaLayers[0];	// mostly useless single-layer case
		}
		
		if (aLayersToShow) {
			aLayersToShow.forEach(function (lay) {
				lay.trigger(); // will show this one and hide its siblings
			});
		}
	}
	

	return {
		about:			"$$$/private/animal/Behavior/HeadTurner/About=Head Turner, copyright 2015.",
		description:	"$$$/animal/Behavior/HeadTurner/Desc=Switch between front, three-quarter, and profile views by turning your head",
		uiName:  		"$$$/animal/Behavior/HeadTurner/UIName=Head Turner",
		defaultArmedForRecordOn: true,

		// NOTE: tags are defined in AdobeFaceTracker.js
		
		defineParams : function () {
			return [
				{ id: "cameraInput", type: "eventGraph", uiName: "$$$/animal/Behavior/HeadTurner/Parameter/cameraInput=Camera Input",
				 	inputKeysArray: ["Head/InputEnabled", "Head/Orient/Y", "Head/Orient/X"],
					uiToolTip: "$$$/animal/Behavior/HeadTurner/param/cameraInput/tooltip=Analyzed head-turn angle from the camera",
					defaultArmedForRecordOn: true,
				 	supportsBlending: true
				},
				{
					id: "layer0", type: "layer", uiName: "$$$/animal/Behavior/HeadTurner/Parameter/leftProfile=Left Profile",
					dephault: { match: "//Adobe.Face.LeftProfile" },
					uiToolTip: "$$$/animal/Behavior/HeadTurner/param/leftProfile/tooltip=Layer showing left side of character"
				},
				{
					id: "layer1", type: "layer", uiName: "$$$/animal/Behavior/HeadTurner/Parameter/leftQuarter=Left Quarter",
					dephault: { match: "//Adobe.Face.LeftQuarter" },
					uiToolTip: "$$$/animal/Behavior/HeadTurner/param/leftQuarter/tooltip=Layer showing left 3/4-view of character"
				},
				{
					id: "layer2", type: "layer", uiName: "$$$/animal/Behavior/HeadTurner/Parameter/frontal=Frontal",
					dephault: { match: "//Adobe.Face.Front" },
					uiToolTip: "$$$/animal/Behavior/HeadTurner/param/front/tooltip=Layer showing front of character"
				},
				{
					id: "layer3", type: "layer", uiName: "$$$/animal/Behavior/HeadTurner/Parameter/rightQuarter=Right Quarter",
					dephault: { match: "//Adobe.Face.RightQuarter" },
					uiToolTip: "$$$/animal/Behavior/HeadTurner/param/rightThreeQuarter/tooltip=Layer showing right 3/4-view of character"
				},
				{
					id: "layer4", type: "layer", uiName: "$$$/animal/Behavior/HeadTurner/Parameter/rightProfile=Right Profile",
					dephault: { match: "//Adobe.Face.RightProfile" },
					uiToolTip: "$$$/animal/Behavior/HeadTurner/param/rightProfile/tooltip=Layer showing right side of character"
				},
				{
					id: "upwardlayer", type: "layer", uiName: "$$$/animal/Behavior/HeadTurner/Parameter/upward=Upward",
					dephault: { match: "//Adobe.Face.Upward" },
					uiToolTip: "$$$/animal/Behavior/HeadTurner/param/upward/tooltip=Layer showing front of character looking upward"
				},
				{
					id: "downwardlayer", type: "layer", uiName: "$$$/animal/Behavior/HeadTurner/Parameter/downward=Downward",
					dephault: { match: "//Adobe.Face.Downward" },
					uiToolTip: "$$$/animal/Behavior/HeadTurner/param/downward/tooltip=Layer showing front of character looking downward"
				},
				{ id: "sensitivity", type: "slider", uiName: "$$$/animal/Behavior/HeadTurner/Parameter/sensitivity=Sensitivity", uiUnits: "%", min: 0, max: 1000, dephault: 100,
					uiToolTip: "$$$/animal/Behavior/HeadTurner/Parameter/sensitivity/tooltip=Affects how far you have to turn your head to switch to another view"
				},
			];
		},

		defineTags: function () {
			return {
				aTags: [
					{
						id: "Adobe.Face.LeftProfile",
						artMatches: ["left profile"],
						uiName: "$$$/animal/Behavior/Face/TagName/LeftProfile=Left Profile",
						tagType: "layertag",
						uiGroups: [{ id:"Adobe.TagGroup.HeadTurn", subSort:0}]				
					},
			
					{
						id: "Adobe.Face.LeftQuarter",
						artMatches: ["left quarter"],
						uiName: "$$$/animal/Behavior/Face/TagName/LeftQuarter=Left Quarter",
						tagType: "layertag",
						uiGroups: [{ id:"Adobe.TagGroup.HeadTurn", subSort:0}]				
					},
			
					{
						id: "Adobe.Face.Front",
						artMatches: ["frontal"],
						uiName: "$$$/animal/Behavior/Face/TagName/Frontal=Frontal",
						tagType: "layertag",
						uiGroups: [{ id:"Adobe.TagGroup.HeadTurn", subSort:0}]				
					},
			
					{
						id: "Adobe.Face.RightQuarter",
						artMatches: ["right quarter"],
						uiName: "$$$/animal/Behavior/Face/TagName/RightQuarter=Right Quarter",
						tagType: "layertag",
						uiGroups: [{ id:"Adobe.TagGroup.HeadTurn", subSort:0}]				
					},
			
					{
						id: "Adobe.Face.RightProfile",
						artMatches: ["right profile"],
						uiName: "$$$/animal/Behavior/Face/TagName/RightProfile=Right Profile",
						tagType: "layertag",
						uiGroups: [{ id:"Adobe.TagGroup.HeadTurn", subSort:0}]				
					},

					{
						id: "Adobe.Face.Upward",
						artMatches: ["upward"],
						uiName: "$$$/animal/Behavior/Face/TagName/Upward=Upward",
						tagType: "layertag",
						uiGroups: [{ id:"Adobe.TagGroup.HeadTurn", subSort:0}]				
					},

					{
						id: "Adobe.Face.Downward",
						artMatches: ["downward"],
						uiName: "$$$/animal/Behavior/Face/TagName/Downward=Downward",
						tagType: "layertag",
						uiGroups: [{ id:"Adobe.TagGroup.HeadTurn", subSort:0}]				
					},
				]
			};			
		},
		
		onCreateBackStageBehavior : function (/*self*/) {
			return { order: 0.2, importance : 0.0 };
		},

		onCreateStageBehavior : function (self, args) {
			var aaLayers = getActiveLayerArrays(args),
				bHideSiblings = true;
			
			self.aaLayers = aaLayers.slice(); // copy array, then add up/down before setting all as triggerable
			
			self.aUpwardLayers = args.getStaticParam("upwardlayer");
			aaLayers.push(self.aUpwardLayers);
			
			self.aDownwardLayers = args.getStaticParam("downwardlayer");
			aaLayers.push(self.aDownwardLayers);
			
			aaLayers.forEach(function (aLays) {
				aLays.forEach(function (lay) {
					lay.setTriggerable(bHideSiblings);
				});
			});
		},

		onAnimate : function (self, args) {
			var headAngle = args.getParamEventValue("cameraInput", "Head/Orient/Y", null, null, true) || 0,
				headUpDownAngle = args.getParamEventValue("cameraInput", "Head/Orient/X", null, null, true) || 0;
			
			// TODO: ask Jim: should this be in an "else" clause for headAngle not being undefined?
			if (args.getParamEventValue("cameraInput", "Head/InputEnabled", null, false)) {
				args.setEventGraphParamRecordingValid("cameraInput");
			}
			
			var sensitivity = args.getParam("sensitivity") * 0.01;
			
			// negated so that it mirror-views by default (i.e. your character looks like it's in the mirror); could be an option
			var adjustedHeadAngle = -headAngle * sensitivity,
				adjustedUpDownAngle = headUpDownAngle * sensitivity;
			
			chooseLayerBasedOnAngle(self, args, adjustedHeadAngle, adjustedUpDownAngle);
		}

	};
});
